// Copyright  2002 Charlie Poole. All rights reserved.
using System;
using System.IO;
using System.Collections;
using EnvDTE;

namespace NUnit.Wizards
{
	public enum TestFixtureType
	{
		ClassSpecificFixture,
		SampleFixture,
		EmptyFixture
	}

	/// <summary>
	/// Summary description for Class1.
	/// </summary>
	public class TestFixtureGenerator
	{
		private TestFixtureType type;
		private string fixtureClassName;
		private CodeClass targetClass;

		private bool setupMethod = false;
		private bool teardownMethod = false;
		private string setupMethodName = "Setup";
		private string teardownMethodName = "TearDown";

		private string testCasePrefix;
		private string testCaseSuffix;

		private string projectName, itemName;
		private string localDirectory, installDirectory;

		private ProjectItems projectItems;
		private Project project;
		private TextWriter writer;

		int nMethods = 0;
		int indent = 1;

		public TestFixtureGenerator( string projectName, 
									 ProjectItems projectItems, 
									 string localDirectory, 
									 string itemName, 
									 string installDirectory )
		{
			this.projectName = projectName;
			this.projectItems = projectItems;
			this.project = (Project) projectItems.Parent;
			this.localDirectory = localDirectory;
			this.itemName = itemName;
			this.installDirectory = installDirectory;

			this.type = TestFixtureType.EmptyFixture;
		}

		public TestFixtureType FixtureType
		{
			get { return type; }
			set { type = value; }
		}

		public CodeClass TargetClass
		{
			get { return targetClass; }
			set { targetClass = value; }
		}

		public string Name
		{
			get { return fixtureClassName; }
			set { fixtureClassName = value; }
		}

		public bool SetupMethod
		{
			get { return setupMethod; }
			set { setupMethod = value; }
		}

		public bool TeardownMethod
		{
			get { return teardownMethod; }
			set { teardownMethod = value; }
		}

		public string SetupMethodName
		{
			get { return setupMethodName; }
			set { setupMethodName = value; }
		}

		public string TeardownMethodName
		{
			get { return teardownMethodName; }
			set { teardownMethodName = value; }
		}

		public string TestCasePrefix
		{
			get { return testCasePrefix; }
			set { testCasePrefix = value; }
		}

		public string TestCaseSuffix
		{
			get { return testCaseSuffix; }
			set { testCaseSuffix = value; }
		}

		public wizardResult Generate()
		{
			try
			{
				AddNUnitReferenceToProject();

				string filePath = Path.Combine( localDirectory, itemName );
				using( this.writer = new StreamWriter( filePath ) )
				{
					GenerateCSharpFile();
				}

				ProjectItem projectItem = projectItems.AddFromFile( filePath );
				Window itemWindow = projectItem.Open( Constants.vsViewKindCode );
				itemWindow.Activate();

				return wizardResult.wizardResultSuccess;
			}
			catch
			{
				return wizardResult.wizardResultFailure;
			}
		}

		private void GenerateCSharpFile()
		{
			string rootNamespace = GetRootNamespace();
			string className = MakeLegalClassName( itemName );
			indent = 0;

			WriteLine( "using System;" );
			WriteLine( "using NUnit.Framework;\n" );
			WriteLine( "namespace " + rootNamespace );
			WriteLine( "{" );

			++indent;

			WriteLine( "[TestFixture]" );
			WriteLine( "public class " + className );
			WriteLine( "{" );

			++indent;

			if ( SetupMethod )
				InsertSetupMethod();

			if ( TeardownMethod )
				InsertTeardownMethod();

			switch ( type )
			{
				case TestFixtureType.ClassSpecificFixture:
					InsertClassSpecificTests();
					break;
				case TestFixtureType.SampleFixture:
					InsertSampleTests();
					break;
				case TestFixtureType.EmptyFixture:
				default:
					InsertTestPlaceholder();
					break;
			}

			--indent;

			WriteLine( "}" );

			--indent;
				
			WriteLine( "}" );
		}

		private void AddNUnitReferenceToProject()
		{
			VSLangProj.VSProject vsProject;
			vsProject = (VSLangProj.VSProject)project.Object;
			vsProject.References.Add( "nunit.framework.dll" );
		}

		private string GetRootNamespace()
		{
			Property prop = project.Properties.Item( "RootNamespace" );

			return prop.Value as string;
		}

		private string MakeLegalClassName( string fileName )
		{
			string className = Path.GetFileNameWithoutExtension( fileName );

			char[] chrClassName = className.ToCharArray(0, className.Length);
			bool modified = false;
			for (int iIndex = 0; iIndex < chrClassName.Length; iIndex++)
			{
				if ((((chrClassName[iIndex] >= 'a') && (chrClassName[iIndex] <= 'z')) || ((chrClassName[iIndex] >= 'A') && (chrClassName[iIndex] <= 'Z')) 
					|| ((chrClassName[iIndex] >= '0') && (chrClassName[iIndex] <= '9')) || (chrClassName[iIndex] == '_')) == false )
				{
					chrClassName[iIndex] = '_';
					modified = true;
				}
			}
			if ( modified )
				className = new string( chrClassName );

			return className;
		}

		private void InsertSetupMethod()
		{
			InsertMethod( "SetUp", SetupMethodName, "Insert setup code here" );
		}

		private void InsertTeardownMethod()
		{
			InsertMethod( "TearDown", TeardownMethodName, "Insert teardown code here" );
		}

		void InsertClassSpecificTests()
		{
			Hashtable names = new Hashtable();

			foreach( CodeElement elem in targetClass.Members )
			{
				if ( elem.Kind == vsCMElement.vsCMElementFunction )
				{
					CodeFunction function = (CodeFunction) elem;
					if ( function.Access == vsCMAccess.vsCMAccessPublic &&
						!names.ContainsKey( function.Name ) )
					{
						if ( function.FunctionKind == vsCMFunction.vsCMFunctionConstructor )
							InsertTestMethod( "Construction" );
						else if ( function.FunctionKind== vsCMFunction.vsCMFunctionFunction )
							InsertTestMethod( function.Name );

						names.Add( function.Name, true );
					}
				}
			}
		}

		void InsertSampleTests()
		{
			InsertTestMethod( "SomeMethod" );
			InsertTestMethod( "AnotherMethod" );
		}

		void InsertTestPlaceholder()
		{
			if ( nMethods > 0 )
				WriteLine();

			WriteLine( "//" );
			WriteLine( "// ToDo: Insert your tests here" );
			WriteLine( "//" );
		}

		private void InsertTestMethod( string methodName )
		{
			InsertMethod( "Test", testCasePrefix + methodName + testCaseSuffix, "Insert test code here"  );
		}

		private void InsertMethod( string attribute, string methodName, string todoComment )
		{
			if ( nMethods > 0 ) WriteLine();

			WriteLine( "[" + attribute + "]" );

			WriteLine( "public void " + methodName + "()" );
			WriteLine( "{" );	
			
			if ( todoComment != null )
			{
				++indent;	
				WriteLine( "// ToDo: " + todoComment );
				--indent;
			}
			
			WriteLine( "}" );

			++nMethods;
		}

		private void WriteLine( string text )
		{
			for ( int i = 0; i < indent; i++ )
				writer.Write( '\t' );

			writer.WriteLine( text );
		}

		private void WriteLine()
		{
			writer.WriteLine();
		}
	}
}
